<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/db.php';

// Set page title
$pageTitle = "Class-wise Collection Report";

// Default academic year (current)
$academic_year = date('Y') . '-' . (date('Y') + 1);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $academic_year = $_POST['academic_year'];
}

// Get class-wise collection
$sql = "SELECT c.class_id, c.class_name,sec.section_name, fs.total_fee, COUNT(DISTINCT s.student_id) as total_students, COUNT(DISTINCT CASE WHEN ft.student_id IS NOT NULL THEN s.student_id END) as paid_students,SUM(ft.amount_paid) as total_collected,COUNT(ft.transaction_id) as transaction_count
    FROM classes c
    JOIN sections sec ON c.section_id = sec.section_id
    JOIN fee_structure fs ON c.class_id = fs.class_id AND fs.academic_year = ?
    LEFT JOIN students s ON c.class_id = s.class_id AND s.status = 'active'
    LEFT JOIN fee_transactions ft ON s.student_id = ft.student_id AND ft.academic_year = ?
    GROUP BY c.class_id, fs.total_fee
    ORDER BY sec.section_id, c.class_name";

$stmt = $conn->prepare($sql);
$stmt->bind_param('ss', $academic_year, $academic_year);
$stmt->execute();
$result = $stmt->get_result();
$class_collections = $result->fetch_all(MYSQLI_ASSOC);

// Calculate totals
$totals = [
    'total_students' => array_sum(array_column($class_collections, 'total_students')),
    'paid_students' => array_sum(array_column($class_collections, 'paid_students')),
    'total_collected' => array_sum(array_column($class_collections, 'total_collected')),
    'transaction_count' => array_sum(array_column($class_collections, 'transaction_count')),
    'total_expected' => array_sum(array_map(function($item) {
        return $item['total_fee'] * $item['total_students'];
    }, $class_collections))
];

// Get all academic years for dropdown
$sql_years = "SELECT DISTINCT academic_year FROM fee_structure ORDER BY academic_year DESC";
$years_result = $conn->query($sql_years);
$years = $years_result->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <?php include '../includes/header-internal.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    
    <div class="container-fluid">
        <div class="row">
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Class-wise Collection Report</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <button class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                            <i class="fas fa-print"></i> Print Report
                        </button>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <div class="col-md-6">
                                <label for="academic_year" class="form-label">Academic Year</label>
                                <select class="form-select" id="academic_year" name="academic_year">
                                    <?php foreach ($years as $year): ?>
                                    <option value="<?= $year['academic_year'] ?>" <?= $year['academic_year'] == $academic_year ? 'selected' : '' ?>>
                                        <?= $year['academic_year'] ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary">Generate Report</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Summary</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-3 mb-3">
                                <div class="card text-white bg-primary">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Expected</h6>
                                        <p class="card-text fs-4">₹<?= number_format($totals['total_expected'], 2) ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 mb-3">
                                <div class="card text-white bg-success">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Collected</h6>
                                        <p class="card-text fs-4">₹<?= number_format($totals['total_collected'], 2) ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 mb-3">
                                <div class="card text-white bg-warning">
                                    <div class="card-body">
                                        <h6 class="card-title">Collection %</h6>
                                        <p class="card-text fs-4">
                                            <?= $totals['total_expected'] > 0 ? 
                                                number_format(($totals['total_collected'] / $totals['total_expected']) * 100, 2) : '0' ?>%
                                        </p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 mb-3">
                                <div class="card text-white bg-info">
                                    <div class="card-body">
                                        <h6 class="card-title">Paid Students</h6>
                                        <p class="card-text fs-4">
                                            <?= $totals['paid_students'] ?>/<?= $totals['total_students'] ?>
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Collection by Class</h5>
                    </div>
                    <div class="card-body">
                        <canvas id="classChart" height="100"></canvas>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5>Detailed Class-wise Report</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Class</th>
                                        <th>Section</th>
                                        <th>Students</th>
                                        <th>Paid Students</th>
                                        <th>Total Fee</th>
                                        <th>Total Collected</th>
                                        <th>Collection %</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($class_collections as $class): ?>
                                    <?php 
                                    $expected = $class['total_students'] * $class['total_fee'];
                                    $collection_percent = $expected > 0 ? 
                                        ($class['total_collected'] / $expected) * 100 : 0;
                                    ?>
                                    <tr>
                                        <td><?= $class['class_name'] ?></td>
                                        <td><?= $class['section_name'] ?></td>
                                        <td><?= $class['total_students'] ?></td>
                                        <td><?= $class['paid_students'] ?></td>
                                        <td>₹<?= number_format($expected, 2) ?></td>
                                        <td>₹<?= number_format($class['total_collected'], 2) ?></td>
                                        <td>
                                            <div class="progress">
                                                <div class="progress-bar" role="progressbar" 
                                                     style="width: <?= $collection_percent ?>%" 
                                                     aria-valuenow="<?= $collection_percent ?>" 
                                                     aria-valuemin="0" aria-valuemax="100">
                                                    <?= number_format($collection_percent, 2) ?>%
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
    
    <script>
        // Prepare data for chart
        const classLabels = <?= json_encode(array_map(function($c) { 
            return $c['section_name'] . ' - ' . $c['class_name']; 
        }, $class_collections)) ?>;
        
        const classData = {
            labels: classLabels,
            datasets: [{
                label: 'Collected (₹)',
                data: <?= json_encode(array_column($class_collections, 'total_collected')) ?>,
                backgroundColor: 'rgba(75, 192, 192, 0.2)',
                borderColor: 'rgba(75, 192, 192, 1)',
                borderWidth: 1
            }, {
                label: 'Expected (₹)',
                data: <?= json_encode(array_map(function($c) { 
                    return $c['total_students'] * $c['total_fee']; 
                }, $class_collections)) ?>,
                backgroundColor: 'rgba(255, 99, 132, 0.2)',
                borderColor: 'rgba(255, 99, 132, 1)',
                borderWidth: 1
            }]
        };

        // Initialize chart
        document.addEventListener('DOMContentLoaded', function() {
            const ctx = document.getElementById('classChart').getContext('2d');
            new Chart(ctx, {
                type: 'bar',
                data: classData,
                options: {
                    responsive: true,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Amount (₹)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Class'
                            }
                        }
                    }
                }
            });
        });
    </script>
</body>
</html>