<?php
require_once '../includes/auth.php';
require_once '../config/db.php';

// Initialize variables
$error = '';
$success = '';

// Get student ID from URL
$student_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);

if (!$student_id) {
    header('Location: students.php');
    exit();
}

// Get classes for dropdown
$class_sql = "SELECT * FROM classes ORDER BY class_name";
$classes = $conn->query($class_sql)->fetch_all(MYSQLI_ASSOC);

// Get student data
$student_sql = "SELECT * FROM students WHERE student_id = ?";
$stmt = $conn->prepare($student_sql);
$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();
$student = $result->fetch_assoc();

if (!$student) {
    header('Location: students.php');
    exit();
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate and sanitize input
    $first_name = filter_input(INPUT_POST, 'first_name', FILTER_SANITIZE_STRING);
    $last_name = filter_input(INPUT_POST, 'last_name', FILTER_SANITIZE_STRING);
    $class_id = filter_input(INPUT_POST, 'class_id', FILTER_VALIDATE_INT);
    $academic_year = filter_input(INPUT_POST, 'academic_year', FILTER_SANITIZE_STRING);
    $admission_date = filter_input(INPUT_POST, 'admission_date', FILTER_SANITIZE_STRING);
    $contact_number = filter_input(INPUT_POST, 'contact_number', FILTER_SANITIZE_STRING);
    $parent_name = filter_input(INPUT_POST, 'parent_name', FILTER_SANITIZE_STRING);
    $address = filter_input(INPUT_POST, 'address', FILTER_SANITIZE_STRING);
    $status = filter_input(INPUT_POST, 'status', FILTER_SANITIZE_STRING);
    
    // Validate required fields
    if (empty($first_name) || empty($last_name) || empty($class_id) || empty($academic_year) || empty($admission_date)) {
        $error = "Please fill in all required fields.";
    } else {
        // Update student data
        $sql = "UPDATE students SET 
                first_name = ?, last_name = ?, class_id = ?, 
                academic_year = ?, admission_date = ?, contact_number = ?, 
                parent_name = ?, address = ?, status = ?
                WHERE student_id = ?";
        
        $stmt = $conn->prepare($sql);
        $stmt->bind_param(
            "ssissssssi", 
            $first_name, $last_name, $class_id,
            $academic_year, $admission_date, $contact_number,
            $parent_name, $address, $status, $student_id
        );
        
        if ($stmt->execute()) {
            $success = "Student information updated successfully!";
            // Refresh student data
            $student_sql = "SELECT * FROM students WHERE student_id = ?";
            $stmt = $conn->prepare($student_sql);
            $stmt->bind_param("i", $student_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $student = $result->fetch_assoc();
        } else {
            $error = "Failed to update student information. Please try again.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <?php include '../includes/header-internal.php'; ?>
    <title>Edit Student</title>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Edit Student</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="students.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-arrow-left"></i> Back to List
                        </a>
                    </div>
                </div>

                <?php if ($error): ?>
                    <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>
                
                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo $success; ?></div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-body">
                        <form method="post" class="needs-validation" novalidate>
                            <div class="row">
                                <div class="col-md-6">
                                    <h5 class="mb-3">Personal Information</h5>
                                    
                                    <div class="mb-3">
                                        <label for="admission_number" class="form-label">Admission Number</label>
                                        <input type="text" class="form-control" id="admission_number" 
                                               value="<?php echo htmlspecialchars($student['admission_number']); ?>" readonly>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="first_name" class="form-label">First Name *</label>
                                        <input type="text" class="form-control" id="first_name" name="first_name" 
                                               value="<?php echo htmlspecialchars($student['first_name']); ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="last_name" class="form-label">Last Name *</label>
                                        <input type="text" class="form-control" id="last_name" name="last_name" 
                                               value="<?php echo htmlspecialchars($student['last_name']); ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="class_id" class="form-label">Class *</label>
                                        <select class="form-select" id="class_id" name="class_id" required>
                                            <option value="">Select Class</option>
                                            <?php foreach ($classes as $class): ?>
                                                <option value="<?php echo $class['class_id']; ?>"
                                                        <?php echo ($student['class_id'] == $class['class_id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($class['class_name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="academic_year" class="form-label">Academic Year *</label>
                                        <select class="form-select" id="academic_year" name="academic_year" required>
                                            <option value="">Select Academic Year</option>
                                            <?php
                                            $current_year = date('Y');
                                            for ($i = 0; $i < 3; $i++) {
                                                $year = $current_year - $i;
                                                $academic_year = $year . '-' . ($year + 1);
                                                $selected = ($student['academic_year'] == $academic_year) ? 'selected' : '';
                                                echo "<option value='$academic_year' $selected>$academic_year</option>";
                                            }
                                            ?>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="admission_date" class="form-label">Admission Date *</label>
                                        <input type="date" class="form-control" id="admission_date" name="admission_date" 
                                               value="<?php echo htmlspecialchars($student['admission_date']); ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="status" class="form-label">Status *</label>
                                        <select class="form-select" id="status" name="status" required>
                                            <option value="active" <?php echo ($student['status'] == 'active') ? 'selected' : ''; ?>>Active</option>
                                            <option value="inactive" <?php echo ($student['status'] == 'inactive') ? 'selected' : ''; ?>>Inactive</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <h5 class="mb-3">Contact Information</h5>
                                    
                                    <div class="mb-3">
                                        <label for="contact_number" class="form-label">Contact Number</label>
                                        <input type="tel" class="form-control" id="contact_number" name="contact_number" 
                                               value="<?php echo htmlspecialchars($student['contact_number']); ?>">
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="parent_name" class="form-label">Parent/Guardian Name</label>
                                        <input type="text" class="form-control" id="parent_name" name="parent_name" 
                                               value="<?php echo htmlspecialchars($student['parent_name']); ?>">
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="address" class="form-label">Address</label>
                                        <textarea class="form-control" id="address" name="address" rows="3"><?php echo htmlspecialchars($student['address']); ?></textarea>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mt-4">
                                <button type="submit" class="btn btn-primary">Update Student</button>
                                <a href="students.php" class="btn btn-secondary">Cancel</a>
                            </div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
    
    <script>
    // Form validation
    (function () {
        'use strict'
        var forms = document.querySelectorAll('.needs-validation')
        Array.prototype.slice.call(forms)
            .forEach(function (form) {
                form.addEventListener('submit', function (event) {
                    if (!form.checkValidity()) {
                        event.preventDefault()
                        event.stopPropagation()
                    }
                    form.classList.add('was-validated')
                }, false)
            })
    })()
    </script>
</body>
</html> 